/* vs3add.c - Add a password to a vs3fs (steganographic)
 *            filesystem.
 * Copyright (C) 1998 Carl van Schaik (carl@leg.uct.ac.za)
 *                    Paul Smeddle (psmeddle@cs.uct.ac.za)
 * VS3FS - [v]an
 *         [S]chaik
 *         [S]meddle
 *         [S]teganographic [F]ile [S]ystem
 *
 *
 * %Begin-Header%
 * This file may be redistributed under the terms of the GNU Public
 * License.
 * %End-Header%
 *
 * Note, the position of the "crypt_master" block is located by taking
 * (e_three_way_key[0] % (blocks on disk))
 */

/* Usage: vs3add [options] device
 * 
 * vs3add by Carl van Schaik
 * ALL code except where noted, by Carl van Schaik
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>
#include <termios.h>
#include <time.h>
#include <errno.h>
#include <mntent.h>
#include <malloc.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <linux/fs.h>
#include <linux/vs3_fs.h>

#include <crypt_funcs.h>
#include <rand.h>  /* Strong Random Number Generator */
#include <vs3lib.h>

char 	*program_name = "vs3add";
char    *bitarray;
struct vs3_crypt_master cryptmst;
int bluff_blks = VS3_WCPRE_ALLOC;


/* Make the encrypted root inode, and commit it to disk */
static void create_e_root_dir()
{
	int i;
	unsigned int pos, cpos, dpos;
	time_t t;
	struct vs3_inode    inode;
	int    offs = offset/VS3_BLK;
        struct vs3_dir_entry *de;
        char dirblock[VS3_BLK];

	printf("Creating new encrypted root inode\n");
	t = time(NULL);

	inode.i_crypt_id_1 = INODE_CRYPT_ID_1;
        inode.i_crypt_id_2 = INODE_CRYPT_ID_2;
	inode.i_blocks     = 1;
	inode.i_type       = VS3_CRYPT_ROOT;
	inode.i_uid        = sb.s_default_uid;
        inode.i_gid        = sb.s_default_gid;
	inode.i_file_size  = VS3_BLK;
        inode.i_ctime      = t;
        inode.i_atime      = t;
        inode.i_mtime      = t;
	inode.i_link_count = 2;
	inode.i_perms      = 0700;
        for (i=0;i<3;i++)
          inode.i_reserved[i] = genrand();
	inode.i_dir_entries = 2; /* the '.' and '..' dirs are handled by the normal root */

	
	/* Check that we have enough disk space */
	read_block(offs, (char *)&sb);
        if (vs3_verify(sb))
          die("superblock encryption test failed");

	if (sb.s_free_blocks < 2)
	  die("disk full");

        /* Find a position and write the block to disk */
        pos = genrand() % sb.s_block_total;
        while (is_block_used(pos))
          pos = (++pos) % sb.s_block_total;

        /* Create the directory entry */
        memset(dirblock,0,VS3_BLK);
        de = (struct vs3_dir_entry *)(dirblock);

        /* Create the '.' directory */
        de->inode_p = pos;
        de->rec_len = 4+2+2+1; /* 32bits + 16 + 16 + `.' */
        de->name_len = 1;
        strcpy(de->name,".");

        /* Make ".." */
        de = (struct vs3_dir_entry *)(dirblock + de->rec_len);
        de->inode_p = sb.s_root_ptr;
        de->rec_len = 4+2+2+2; /* 32bits + 16 + 16 + `..' */
        de->name_len = 2;
        strcpy(de->name,"..");


	/* Update the crypt-mastor */
        cpos = ((e_three_way_key[0]+e_three_way_key[1]+
                e_three_way_key[2]) % sb.s_block_total);

        read_e_block(cpos, (char *)&cryptmst);
	if ((cryptmst.cm_crypt_1 != SUPER_CRYPT_ID_1) ||
	    (cryptmst.cm_crypt_2 != SUPER_CRYPT_ID_2))
	  die("decryption of encrypted master block failed");
	cryptmst.cm_root_ptr = pos;

	write_e_block(cpos, (char *)&cryptmst);

        /* Update the root inode and write the directory */
        dpos = genrand() % sb.s_block_total;
        while (is_block_used(dpos))
          dpos = (++dpos) % sb.s_block_total;
        set_block_used(dpos);
        set_block_used(pos);

        inode.i_direct[0] = dpos;
        write_e_data_block(pos, (char *)&inode);
        write_e_data_block(dpos, (char*)&dirblock);
}


/* (Fast) Checks if a specified block is used or not */
int f_is_block_used(unsigned int blk)
{
        char   bitm[VS3_BLK];
        int offs = offset/VS3_BLK;
        unsigned int i,o;

        i = (blk-offs) / VS3_GROUP;
        o = (blk-offs) % VS3_GROUP;

        if (i > sb.s_group_no)
          return (1);
        if (blk > sb.s_block_total)
          return (1);

	memcpy((char*)&bitm, (char*)&bitarray[i*VS3_BLK], VS3_BLK);

        return (is_bit_set(bitm, o));
}

/* (fast) Sets a block marked used on the disk */
void f_set_block_used(unsigned int blk)
{
        char   bitm[VS3_BLK];
        int    offs = offset/VS3_BLK;
        unsigned int i,o;

        i = (blk-offs) / VS3_GROUP;
        o = (blk-offs) % VS3_GROUP;

        if (i > sb.s_group_no)
          die("tried to read past end of fs");

	memcpy((char*)&bitm, (char*)&bitarray[i*VS3_BLK], VS3_BLK);

        if (is_bit_set(bitm, o))
          die("tried to set bit that already set, corruption warning!");

        bitset(bitm, o);

	memcpy((char*)&bitarray[i*VS3_BLK],(char*)&bitm,VS3_BLK);

        sb.s_free_blocks--;
}


/* Set a particular block in the winnowing and chaffing */
/* bitmaps to used   */
static void set_wc_block(unsigned int pos)
{
	int offs = offset/VS3_BLK;
	unsigned int i,o;

	i = (pos-offs) / VS3_GROUP;
	o = (pos-offs) % VS3_GROUP;

	if (pos < (96*VS3_GROUP+offs)) /* 768 MB */
	  {
	    read_e_block(cryptmst.i_direct[i], (char*)&bitmap);

	    if (is_bit_set(bitmap, o))
	      die("tried to set bit that already set, corruption warning!");

	    bitset(bitmap, o);

	    write_e_block(cryptmst.i_direct[i], (char*)&bitmap);
	  }
	else
	if (pos < (80*VS3_BLK*VS3_GROUP+offs)) /* 768MB - 640 GB */
	  {
	  }
	else  /* 640GB - 512TB */
	  printf("Hey your disk is bigger than 640GB, you write this\n");
}


/* Create the Cryptographic Master Record */
static void create_crypt_master()
{
	unsigned int i, t;
	unsigned int pos;

	cryptmst.cm_crypt_1 = SUPER_CRYPT_ID_1;
	cryptmst.cm_crypt_2 = SUPER_CRYPT_ID_2;
	cryptmst.cm_wc_blocks = 0;

	read_block(offset/VS3_BLK, (char *)&sb);
	if (vs3_verify(sb))
	  die("superblock encryption test failed");

	pos = ((e_three_way_key[0]+e_three_way_key[1]+
		e_three_way_key[2]) % sb.s_block_total);
	set_block_used(pos);

	for (i=0;i<12;i++)
	  cryptmst.cm_reserved[i] = genrand();

	for (i=0;i<96;i++)
	  cryptmst.i_direct[i] = genrand();
	for (i=0;i<80;i++)
	  cryptmst.i_single[i] = genrand();
	for (i=0;i<64;i++)
	  cryptmst.i_double[i] = genrand();

	read_block(offset/VS3_BLK, (char *)&sb);
	if (vs3_verify(sb))
	  die("superblock encryption test failed");

	memset(bitmap,0,VS3_BLK);

	t = (sb.s_free_blocks/100)*bluff_blks;
 
	for (i=0;i<sb.s_group_no;i++)
	  {
	    pos = genrand() % sb.s_block_total;

	    while (is_block_used(pos))
	      pos = (++pos) % sb.s_block_total;

	    set_block_used(pos);
	    write_e_block(pos, (char*)&bitmap);

 	    if (i < 96)
		cryptmst.i_direct[i] = pos;
	  }

	read_block(offset/VS3_BLK, (char *)&sb);
	if (vs3_verify(sb))
	  die("superblock encryption test failed");

	bitarray = (char*)malloc(sb.s_group_no*VS3_BLK);
	if (bitarray == NULL)
	  {
	    printf("Not enough memory to buffer bitmaps\n");
	    printf("Performing non-cached operation (this may take a while)\n");
	    for (i=0;i<t;i++)
	      {        
		printf("%i of %i \r",i,t);
		fflush(stdout);
	        pos = genrand() % sb.s_block_total;
	        while (is_block_used(pos))
	          pos = (++pos) % sb.s_block_total;

	        set_block_used(pos);
	        set_wc_block(pos);
	      }
	  }
	else
	  {
		for (i=0;i<sb.s_group_no;i++)
		  read_block(offset/VS3_BLK+i*VS3_GROUP+1,(char*)&bitarray[i*VS3_BLK]);

		for (i=0;i<t;i++)
		  {
		    if (i % 256 == 0)
		      {
			printf("\r%i of %i ",i,t);
			fflush(stdout);
		      }
		    pos = genrand() % sb.s_block_total;
		    while (f_is_block_used(pos))
		      pos = (++pos) % sb.s_block_total;

		    f_set_block_used(pos);
		    set_wc_block(pos);
		  }
		for (i=0;i<sb.s_group_no;i++)
		  write_block(offset/VS3_BLK+i*VS3_GROUP+1,(char*)&bitarray[i*VS3_BLK]);

//		for (i=0;i<VS3_GROUP;i++)
//		  printf("%i",f_is_block_used(i+1));

		free(bitarray);
	  }

	write_block(offset/VS3_BLK, (char *)&sb);

	pos = ((e_three_way_key[0]+e_three_way_key[1]+
		e_three_way_key[2]) % sb.s_block_total);
	write_e_block(pos,(char*)&cryptmst);

	sync_super_blocks();
}


/* Checks to see if system password is correct */
/* and whether user password exists already or */
/* cannot be used because of conflict          */
static void check_password()
{
	unsigned int pos,i;
	word32 j,k,l,m;
	char kp;

	read_block(offset/VS3_BLK, (char *)&sb);
	if (vs3_verify(sb))
	  die("superblock encryption test failed");
	if (sb.s_free_blocks < (sb.s_group_no + 2))
	  die("not enough free space on device");

	pos = ((e_three_way_key[0]+e_three_way_key[1]+
		e_three_way_key[2]) % sb.s_block_total);
	read_e_block(pos,(char*)&cryptmst);

	if ((cryptmst.cm_crypt_1 == SUPER_CRYPT_ID_1) &&
	    (cryptmst.cm_crypt_2 == SUPER_CRYPT_ID_2))
	  die("That password already exists on the disk");

	i = 0;
	k = 0;

	if (is_block_used(pos))
	  {
	    while ((text_key[i] != 0) && (i < 128))
	      i++;

	    while(is_block_used(pos) && (k<10000))
	      {
		text_key[i] = 49 + k % 74;
		for (j = i; j < 127; j++)
		  {
		    m = 1;
		    for (l=0;l<(j-i);l++)
			m *= 73;
		    if (m == 0)
			text_key[j+1] = 0;
		    else
		      text_key[j+1] = 49*(((k/m) % 73) != 0) + (((k/m) / 73) % 73);
		  }
		k++;
		make_3way_key(text_key, e_three_way_key);
		make_3way_key(text_key, e_three_way_key);

		pos = ((e_three_way_key[0]+e_three_way_key[1]+
			e_three_way_key[2]) % sb.s_block_total);
		printf("%s %i\n",text_key,pos);
	      }
	    make_idea_key(text_key, e_idea_key);
	    make_3way_key(text_key, e_three_way_key);

	    if (k < 10000)
	      {
	   	printf("Due to disk space issues, your requested key is not available\n");
	   	printf("The system has generated a derivative of your key that is accepable\n");
          	printf("New key : %s\n",text_key);
		if (wants_key)
		    {
		      printf("This is your new private key\n");
		      printf("IDEA KEY = 0x%04x%04x%04x%04x\n3WAY KEY = 0x%08lx%08lx%08lx\n",
		      e_idea_key[0],e_idea_key[1],e_idea_key[2],e_idea_key[3],
		      e_three_way_key[0],e_three_way_key[1],e_three_way_key[2]);
		    }
		printf("Do you want to use this new key? ");
		fflush(stdout);

		kp = getchar();
		if ((kp != 'y') && (kp != 'Y'))
		{
		  printf("\nNo changes were made to %s\n",device_name);
		  clean_up();
		  close(device);
		  exit(0);
		}
	      }
	    else
	      die("disk nearly full, unable to satisfy request: Try another password");
	  }
	else
	  printf("User password accepted\n");
}

/* Displays the programs options
 */
static void show_options()
{
	printf("USAGE:  vs3add [options] <dev>\n");
	printf("	Options : -k      : display encryption keys\n");
	printf("		  -i      : show fs information on completion\n");
	printf("                  -c <num>: percent of free disk space to use for cover\n");
	printf("                            (default is %i%% of disk free)\n",bluff_blks);
}


/* This checks if all the options were gathered properly */
static int check_options()
{
	if (got_device != 1)
	  return(0);

	return(1);
}

/* Process command line arguments */
static void PRS(int argc, char *argv[])
{
	char k,l;

	printf("[0;1;37;33m");
	printf("vs3add v0.0, Steganographic File System\n");
	printf("[0;1;37;00m");
	printf("(C) 1998 Paul Smeddle (psmeddle@cs.uct.ac.za)\n");
	printf("         Carl van Schaik (carl@leg.uct.ac.zan\n");

	argc--;
	argv++;

	if (argc == 0) {
		show_options();
		exit(1);
		}

	while (argc--)
	{
	  if (argv[0][0] == '-')
	    {
top:
	      k = (*(++argv[0])); 
	      k = argv[0][0];
	      l = 1;
	      switch(k)
		{
		  case 'k' :
			{
			  wants_key = 1;
			  break;
		 	}
		  case 'i' :
			{
			  show_stats = 1;
			  break;
			}
		  case 'c' :
		        {
			  if (!argc--)
			    { show_options(); break; }
			  argv++;
			  bluff_blks = atoi(*argv);
			  if (bluff_blks >= 80)
			    {
			      printf("That is too much space to waste : %i%%\n",bluff_blks);
			      exit(0);
			    }
			  l--;
			  break;
		        }
		  default :
			{
			  printf("Invalid Option : %c\n",k);
			  show_options();
			  exit(1);
			}
		}
		if ((l) && (argv[0][1] != '\0'))
		  { goto top; }
	    }
	  else
	    {
	      device_name = *argv;
	      got_device++;
	     }
	  argv++;
	}

  if (!check_options())
    {
      show_options();
      exit(1);
    }
}


/* main procedure */
int main (int argc, char *argv[])
{
	PRS(argc, argv);     /* Process command line options */

	ask_for_key('n');    /* Ask the user for his system Key */
        ask_for_key('e');    /* Ask the user for his new private Key */

	sgenrand(time(NULL)+three_way_key[2]);  /* set seed to time */          

	init_3way();         /* Initialise the 3-Way encryption cipher */

	device = open(device_name,O_RDWR); /* Open read/write */
	if(device < 0) {
	  clean_up();
	  perror(device_name);
	  exit(errno);
	}


	/* Check to see if device is valid etc */
	/* And if the user really want to procede */
	if (check_device())
	{
	  clean_up();
	  close(device);
	  exit(0);
	}

	check_password();

	create_crypt_master();
	create_e_root_dir();

        read_block(VS3_BOOT_SECT_SIZE/VS3_BLK, (char *)&sb);
	if (vs3_verify(sb))
	  die("superblock encryption test failed");

	if (show_stats)
	  get_fs_info();

	clean_up();
	close(device);
	printf("vs3add completed, and passed test\n");
	return 0;
}
